using System;
using System.Data;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Mike Smith</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/17/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///		Data access component implementing 
	///		reagent-related data persistence logic
	///	</summary>

	#endregion

	public class Reagent
	{

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1060"> 
		///		<ExpectedInput>DivisonCode, Expiration Date Ranges, Received Date Ranges</ExpectedInput>
		///		<ExpectedOutput>Data table of reagents</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1940"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		Gets a DataTable of reagents satisfying the search criteria
		/// </summary>
		/// <param name="divisionCode">Division code</param>
		/// <param name="expirationDateBefore">Upper bound for expiration date</param>
		/// <param name="expirationDateAfter">Lower bound for expiration date</param>
		/// <param name="receivedDateBefore">Upper bound for received date</param>
		/// <param name="receivedDateAfter">Lower bound for received date</param>
		/// <returns>DataTable containing reagents data</returns>
		public static DataTable GetReagents(string divisionCode, DateTime expirationDateBefore,	DateTime expirationDateAfter, DateTime receivedDateBefore, DateTime receivedDateAfter)
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetReagents.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetReagents.expirationdatebefore, SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetReagents.expirationdateafter, SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetReagents.receiveddatebefore, SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetReagents.receiveddateafter, SqlDbType.DateTime)
				};

			prms[0].Value = divisionCode;

			if (expirationDateBefore == DateTime.MaxValue)
				prms[1].Value = SqlDateTime.MaxValue.Value.ToShortDateString();
			else
				prms[1].Value = expirationDateBefore;

			if (expirationDateAfter == DateTime.MinValue)
				prms[2].Value =  SqlDateTime.MinValue.Value.ToShortDateString();
			else
				prms[2].Value = expirationDateAfter;

			if (receivedDateBefore == DateTime.MaxValue)
				prms[3].Value =  SqlDateTime.MaxValue.Value.ToShortDateString();
			else
				prms[3].Value = receivedDateBefore;

			if (receivedDateAfter == DateTime.MinValue)
				prms[4].Value = SqlDateTime.MinValue.Value.ToShortDateString();
			else
				prms[4].Value = receivedDateAfter;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagents.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1061"> 
		///		<ExpectedInput>Manufacturer name.</ExpectedInput>
		///		<ExpectedOutput>Data table containing reagent information for a manufacturer within a division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1941"> 
		///		<ExpectedInput>Invalid Manufacturer</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a data table containing reagent information for a manufacturer within a division.
		/// </summary>
		/// <param name="manufacturerName">Manufacturer name</param>
		/// <returns>DataTable containing reagents data for a specified manufacturer</returns>
		public static DataTable GetManufacturerReagentNames(string manufacturerName)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetManufacturerReagentNames.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetManufacturerReagentNames.reagentmanufacturername, SqlDbType.VarChar)
				};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = manufacturerName;

			return Common.StoredProcedure.GetData(STOREDPROC.GetManufacturerReagentNames.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1062"> 
		///		<ExpectedInput>DataTable</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1942"> 
		///		<ExpectedInput>Inavalid DataTable</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts records into the reagent and reagent parameter tables (saves new reagents)
		/// </summary>
		/// <param name="dtReagents">DataTable containing reagents data</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtWorkloadEvents">Data needed to capture workload event records</param>
		/// <returns>True on success. Otherwise - false.</returns>
		public static bool InsertReagents(DataTable dtReagents, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable[] dtArray = new DataTable[2];
			string[] spArray = new string[2];
			//
			dtArray[0] = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
			spArray[0] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
			//
			dtArray[1] = Common.Utility.AppendLastUpdateInformation(dtReagents,lastUpdateFunctionId);
			spArray[1] = STOREDPROC.InsertReagent.StoredProcName;

			return (new StoredProcedure().TransactionalGetValue(spArray, dtArray) == 0);
		}	

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1063"> 
		///		<ExpectedInput>DataTable of Reagents, DataTable of Reagent Changes, UpdateFunctionId</ExpectedInput>
		///		<ExpectedOutput>success</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1944"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates records in the reagent and reagent parameter tables
		/// (saves reagent changes).
		/// </summary>
		/// <param name="dtReagents">DataTable</param>
		/// <param name="dtReagentChanges">DataTable</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtWorkloadEvents">WorkloadEvent data</param>
		/// <returns>True on success. Otherwise - false. </returns>
		public static bool UpdateReagent(DataTable dtReagents, DataTable dtReagentChanges, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();

			if (dtReagents.Rows.Count > 0)
			{
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtReagents,lastUpdateFunctionId));
				sprocArray.Add(STOREDPROC.UpdateReagent.StoredProcName);

				dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtReagentChanges, lastUpdateFunctionId));
				sprocArray.Add(STOREDPROC.InsertReagentChange.StoredProcName);

				if(dtWorkloadEvents != null)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				}

				return (new StoredProcedure().TransactionalGetValue(sprocArray, dtArray) == 0 );
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1064"> 
		///		<ExpectedInput>Valid ReagentId</ExpectedInput>
		///		<ExpectedOutput>DataTable containing the current stock level of the reagent</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1950"> 
		///		<ExpectedInput>Invalid Id</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///		Gets the current stock level of the reagent
		/// </summary>
		/// <param name="reagentTypeId">Reagent type ID</param>
		/// <returns>DataTable containing reagent stock level</returns>
		public static DataTable GetReagentTypeStockLevel(int reagentTypeId)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentTypeStockLevel.reagenttypeid, SqlDbType.Int),
					new SqlParameter(STOREDPROC.GetReagentTypeStockLevel.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = reagentTypeId;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentTypeStockLevel.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1084"> 
		///		<ExpectedInput>Valid division and ReagentType id.</ExpectedInput>
		///		<ExpectedOutput>DataTable of reagents for a reagent type and division.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1953"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>Empty Datatable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a DataTable of reagents for a reagent type and division.
		/// </summary>
		/// <param name="divisionCode">Division code</param>
		/// <param name="reagentTypeId">Reagent type ID</param>
		/// <returns>DataTable containing reagents data</returns>
		public static DataTable GetReagentsByType(string divisionCode, int reagentTypeId)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentsByType.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetReagentsByType.reagenttypeid, SqlDbType.Int)
				};

			prms[0].Value = divisionCode;
			prms[1].Value = reagentTypeId;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentsByType.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1288"> 
		///		<ExpectedInput>Rack Guid</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing the list of reagent lot numbers for retested rack</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1955"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the list of reagent lot numbers for retested rack 
		/// </summary>
		/// <param name="rackGuid">Rack GUID</param>
		/// <returns>DataTable containing reagent lots data</returns>
		public static DataTable GetReagentLotsForRetest(Guid rackGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentLotsForRetest.RackGuid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetReagentLotsForRetest.divisioncode, SqlDbType.Char)
				};

			prms[0].Value = rackGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentLotsForRetest.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1295"> 
		///		<ExpectedInput>Rack name</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing the list of reagent lot numbers from previous test for a tested rack</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1956"> 
		///		<ExpectedInput>Invalid RackName</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the list of reagent lot numbers from previous test for a tested rack
		/// </summary>
		/// <param name="rackName">Rack name</param>
		/// <returns>DataTable containing reagent lot numbers</returns>
		public static DataTable GetReagentLotsFromPreviousTest(string rackName)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentLotsFromPreviousTest.rackname, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetReagentLotsFromPreviousTest.divisioncode, SqlDbType.VarChar)

				};

			prms[0].Value = rackName;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentLotsFromPreviousTest.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1497"> 
		///		<ExpectedInput>Rack Guid, ReagentType Id, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>The DataTable of Reagent Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1498"> 
		///		<ExpectedInput>Invlaid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the reagent lot number used for a tested rack
		/// </summary>
		/// <param name="RackGuid">Rack GUID</param>
		/// <param name="reagentTypeId">Reagent Type ID</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns>DataTable containing reagent lot number</returns>
		public static DataTable GetReagentFromRack(Guid RackGuid, int reagentTypeId, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetReagentFromRack.RackGuid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetReagentFromRack.reagenttypeid, SqlDbType.Int),
				new SqlParameter(STOREDPROC.GetReagentFromRack.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = RackGuid;
			prms[1].Value = reagentTypeId;
			prms[2].Value = divisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentFromRack.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1501"> 
		///		<ExpectedInput>Invoice Number</ExpectedInput>
		///		<ExpectedOutput>The DataTable of Reagents for an invoice</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1502"> 
		///		<ExpectedInput>Invalid Invoice</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the list of reagents for the invoice
		/// </summary>
		/// <param name="invoiceNumber">Invoice number</param>
		/// <returns>DataTable containing list of reagents</returns>
		public static DataTable GetReagentsForInvoice(string invoiceNumber)
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetReagentsForInvoice.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetReagentsForInvoice.invoicenumber, SqlDbType.VarChar)
				};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = invoiceNumber;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentsForInvoice.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1503"> 
		///		<ExpectedInput>Reagent Guid</ExpectedInput>
		///		<ExpectedOutput>The DataTable Reagent Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1506"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		///	Retrieves reagent data
		///</summary>
		///<param name="reagentGuid">Reagent GUID</param>
		///<returns>DataTable containing reagent data</returns>
		public static DataTable GetReagentByGuid(Guid reagentGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentByGuid.reagentguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetReagentByGuid.divisioncode, SqlDbType.Char)
				};

			prms[0].Value = reagentGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentByGuid.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1507"> 
		///		<ExpectedInput>Reagent Guid</ExpectedInput>
		///		<ExpectedOutput>The DataTable of previous rack lot entries</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1516"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function, returns all previous rack lot entries for a supplied
		/// reagent, used to determine if a testing phase has ever been used on the reagent before
		/// BR_41.26
		/// </summary>
		/// <param name="reagentGuid">Reagent unique identifier</param>
		/// <returns>Rack Lot data table</returns>
		public static DataTable GetPreviousTestingPhases(Guid reagentGuid)
		{
			if (reagentGuid == Guid.Empty)
			{
				throw new Common.DataAccessLayerException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Reagent GUID").ResString);
			}
			
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetReagentPreviousAntigenTypingPhases.reagentguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetReagentPreviousAntigenTypingPhases.divisioncode, SqlDbType.Char)
				};
			
			prms[0].Value = reagentGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetReagentPreviousAntigenTypingPhases.StoredProcName, prms).Tables[0];
			
		}
	}
}
